import numpy as np

import numpy as np

import numpy as np

def simplex_mu_dense(K, d, radii=None, rng=None):
    """
    在 d 维空间构造 K 个等角向量 (regular simplex 顶点)，
    并通过随机正交变换让每个向量至少有 K+1 个非零分量。

    参数:
    - K: 类别数
    - d: 目标维度 (>= K-1)
    - radii: None 或 长度 K 的 array, 每个向量的放缩因子
    - rng: np.random.Generator 或 None

    返回:
    - mu: (K, d) 的 ndarray
    """
    if rng is None:
        rng = np.random.default_rng()
    if d < K-1:
        raise ValueError("需要 d >= K-1")

    # Step 1: 构造标准 simplex (K x (K-1))
    S = np.eye(K) - np.ones((K, K)) / K
    vals, vecs = np.linalg.eigh(S)
    idx = np.argsort(vals)[::-1][:K-1]
    basis = vecs[:, idx]             # K x (K-1)
    coords = S @ basis
    coords = coords / np.linalg.norm(coords, axis=1, keepdims=True)

    # Step 2: 随机正交变换，嵌入到 R^d
    A = rng.normal(size=(d, K-1))
    Q, _ = np.linalg.qr(A)           # d x (K-1)，正交列
    mu = coords @ Q.T                # (K, d)

    # Step 3: 半径缩放
    if radii is not None:
        radii = np.asarray(radii)
        if radii.shape[0] != K:
            raise ValueError("radii 长度需为 K")
        mu = mu * radii[:, None]

    return mu



# ===== 测试 =====
for K in [3, 4, 5]:
    d = K+1
    U = simplex_mu_dense(K,d)
    G = U @ U.T  # Gram matrix
    print(f"K={K}, 向量维度={U.shape[1]}")
    print("Gram 矩阵 (前5行):")
    print(np.round(G, 3))
    print()
